/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/  

/**
 * \file    eeprom_test.c
 *
 * \brief   Support routines for I2C EEPROM tests
 *
 * This file contains the support routines for configuring and
 * accessing the I2C based Serial EEPROM.
 *
 * \author  0043
 *
 * \version 01a,13aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "stdio.h"
#include "DM814x_i2c.h"
#include "DM814x_EVM.h"
#include "DM814x_types.h"
#include "eeprom.h"


/*
 *====================
 * Function implementations
 *====================
 */
 /** **************************************************************************
 * \n \brief Routine to test EEPROM I2C 
 *
 * This routine tests EEPROM by Writing 1 page of EEPROM and reading the same
 * then both read and written datas are compared .Returns 0 on Successfull 
 * comparison.
 * 
 * Testing Restricted to one page to reduce time.
 * 
 * \param Power_cycle_test   [IN]      Testid
 *
 * \return
 * \n      return SUCCESS for success  - Description
 * \n      return FAILED for error   - Description
 */

STATUS eeprom_test(INT16 power_cycle_test)
{
	STATUS u32RetVal = 0;
	INT16 i, j;
    UINT8 *p8;
    UINT32 src, dst;
    static UINT8 tx[4][EEPROM_PAGE_SIZE];
    static UINT8 rx[4][EEPROM_PAGE_SIZE];    
    INT16 errors = 0;
    UINT16 pages = 1;
    UINT16 page_size =4;
 
    u32RetVal = DM814xI2cInit(EEPROM_I2C_BUS_NUM);
	
    /* Create the test pattern */
    p8 = ( UINT8* )tx;
    for ( i = 0 ; i < pages * page_size ; i++ )
        *p8++ = i;

    /* Clear RX buffer */
    p8 = ( UINT8* )rx;
    for ( i = 0 ; i < pages * page_size ; i++)
        *p8++ = 0;

    if (!power_cycle_test)
    {
        /* Write 1 page at a time */
	for ( i = 0 ; i < pages ; i++ )
    	{
            src = ( UINT32 )&tx[i];
    	    dst = i * page_size;

    	    if ( EEPROM_write( src, dst, page_size ) )
			{
				return 1;
			}
			else
			{
			printf ("Successfully writen to EEPROM.\r\n");
			}

    	    DM814x_usecDelay( 25000 );
    	}
    }

    /* Read 1 page at a time */
    for ( i = 0 ; i < pages ; i++ )
    {
        src = i * page_size;
        dst = ( UINT32 )&rx[i];

		printf ("Reading back the contents.\r\n");
        if ( EEPROM_read( src, dst, page_size ) )
		{
            return 2;
		}
		else
		{
			printf ("Successfully read the content back. Verifying it.\r\n");
		}

        DM814x_usecDelay( 25000 );
    }

    /* Verify tx & rx */
    for ( i = 0 ; i < pages ; i++ )
    {
        errors = 0;

        for ( j = 0 ; j < page_size ; j++ )
            if ( tx[i][j] != rx[i][j] )
                errors++;

        if ( errors == 0 )
        {
            printf( "    Page %d: PASS\n", i );
        }
        else
        {
        	u32RetVal = FAILED;
            printf( "    Page %d: FAIL\n", i );
        }
    }

    return u32RetVal;
}
